## Overview

Library for working with API 3DiVi BAF.

## Use cases

-   Use BAF API with collecting fingerprint
-   Use Platform API
-   Use LRS API

## Installation

The component is supplied as a TGZ archive. Move the archive to your project's root folder and add the following line to your package.json in the dependencies section:

```
"tdvc": "file:tdvc-{version}.tgz"
```

Import the library.

```javascript
import API from 'tdvc';
```

## Initialization

To use the library, create an instance of the API module. The module accepts various tokens as configuration. Below is a description of the configuration:

**baseUrl** (string) - URL address of 3DiVi BAF. Required.

**bafToken** (string) - BAF Web API Secret Key.

**platformToken** (string) - Platform service token.

**lrsToken** (string) - Liveness Reflection service token.

### Initialization examples

```javascript
const api = new API({
    baseUrl: 'https://baf.example.com',
    bafToken: 'sk_00000000-0000-0000-0000-000000000000',
    platformToken: '00000000-0000-0000-0000-000000000000',
    lrsToken: '00000000-0000-0000-0000-000000000000',
});
```

## API Docs

This section describes the APIs available in this library.

> **_ATTENTION_:** You can learn more about the types returned and accepted during development when going to d.ts files.

### BAF

#### getApplicantByFilter

> **getApplicantByFilter** (**_page_**: number = 1, **_pageSize_**: number = 10, **_textFilter_**?: string) => Promise<AxiosResponse<**TGetApplicantsList**, any>> - Method for obtaining a list of applicants. Accepts the following parameters as input: page (default 1), pageSize (default 10) and textFilter. Returns a list of applicants.

#### getApplicantByID

> **getApplicantByID** (**_applicantId_**: string) => Promise<AxiosResponse<**TApplicant**, any>> - Method for obtaining an applicant by his ID. Takes the applicant's ID as a parameter. Returns the applicant, if available.

#### createApplicant

> **createApplicant** (**_data_**: [string, FormDataEntryValue][]) => Promise<AxiosResponse<**TCreateApplicantResult**, any>> - Method for obtaining the creation of an applicant. Accepts fields for creating an applicant as a parameter. To simply get these fields, we recommend calling the FormData.entries() method. Returns information about the created applicant.

#### authorizeApplicant

> **authorizeApplicant** (**_applicantId_**: string, **_faceImageBase64_**: string, **_endeavorId_**?: string) => Promise<AxiosResponse<**TValidateApplicantResult**, any>> - Method for authorizing the applicant. It takes as parameters: ID of the applicant who needs to be authorized, faceImageBase64 - image with the applicant’s face, endeavorId - attempt ID from the LRS service. Returns the result of authorization checks.

#### registerApplicant

> **registerApplicant** (**_applicantId_**: string, **_faceImageBase64_**: string, **_endeavorId_**?: string) => Promise<AxiosResponse<**TRegisterApplicantResult**, any>> - Method for registering an applicant. It takes as parameters: ID of the applicant who needs to be registered, faceImageBase64 - image with the applicant’s face, endeavorId - attempt ID from the LRS service. Returns the result of registration checks.

### Platform

#### processImage

> **processImage** (**_base64Image_**: string, **_fields_**: TProcessImageFields) => Promise<AxiosResponse<**TProcessImageResult**, any>> - Method for image processing. It takes as parameters: base64Image - the image for processing, fields - a set of fields that need to be calculated. Returns the result of processing.

### LRS

#### getLRSOffer

> **getLRSOffer** (**_offer_**: RTCSessionDescription | undefined | null, **_endeavorId_**: string) => Promise<AxiosResponse<**RTCSessionDescriptionInit**, any>> - Method for receiving SDP Offer for RTCP Connection.

#### getLRSEndeavor

> **getLRSEndeavor** (**_applicantId_**: string) => Promise<AxiosResponse<**TEndeavor**, any>> - Method for creating an attempt session in LRS.

#### getLRSCredentials

> **getLRSCredentials** () => Promise<AxiosResponse<**string**, any>> - Method for obtaining connection data to the STUN server.

## Examples

An example of library initialization and processImage request with result processing.:

```javascript
const api = new API({
  baseUrl: 'https://baf.example.com',
  platformToken: '00000000-0000-0000-0000-000000000000',
});

const b64 = '/9j/4AAQSkZJRgABAQEAYABgAAD...';

api.processImage(b64, ['quality', 'liveness'])
  .then((response) => {
    response.data;
  })
  .then((data) => {
    console.log(
      `Liveness result: ${data.data.processImage.objects[0].liveness.value}`
    );
    console.log(
      `Quality result: ${data.data.processImage.objects[0].quality.totalScore}`
    );
  }).catch ((error) => {
    const responseStatus = (error as AxiosError).response?.status;
    if (responseStatus && responseStatus >= 500) {
      console.log('Server error code is greater than 500');
    } else if (responseStatus && responseStatus >= 400) {
      console.log('Server error code is less than 400');
      const { code } = ((error as AxiosError).response?.data as { code:string });
      const { message } = ((error as AxiosError).response?.data as { message:string });
      console.log(`Error code: ${code}. Error message: ${message}`);
    }
  });
```

## Version history

-   **1.6.1**

    - Changed the format of the request to search for an applicant by applicantFields values

-   **1.6.0**

    - Expanded the set of data to be collected in the device fingerprint

-   **1.5.6**

    -   Added ability to send a cancel request signal to getLRSOffer

-   **1.5.5**

    -   Added an optional customSettings parameter to request validation

-   **1.5.4**

    -   Updated the return value types for the attempt validation request

-   **1.5.3**

    -   Fixed a bug that caused not all available data in the fingerprint to be sent during registration or authorization.

-   **1.5.2**

    -   Add requests:
        1. getRNDActionPatter
        2. saveLivenessReflectionResult
        3. saveLivenessActiveResult

-   **1.5.1**

    -   Add timeout to getting device fingerprint.

-   **1.5.0**

    -   Fix **_API Manager_** configuration. Now each of the tokens is optional. Rename **_secretKey_** to **_bafToken_**.
    -   Add configuration validations. If a method is used for which the token was not passed in the configuration, a **_Configuration error_** will be thrown.
    -   Update API methods:
        -   Add **_page_** and **_pageSize_** parameters in **_getApplicantByFilter_** method. Now **_textFilter_** parameter is optional.
        -   Now **_endeavorId_** parameter is optional in **_authorizeApplicant_** and **_registerApplicant_** methods
        -   In the **_processImage_** method you can now pass a list of fields that need to be calculated. The method also checks for duplicate fields.
        -   The **_getLRSOffer_** method now returns an **_RTCSessionDescriptionInit_** type.
        -   The return types in the **_authorizeApplicant_** and **_registerApplicant_** methods have been updated and the **_hasRiskEvents_** field has been added.
    -   Update README file. Add API Docs and using examples.

-   **1.4.0**

    -   Update API types
    -   Fix Fingerprint collecting

-   **1.3.1**

    -   Update API types
    -   Add Liveness Reflection results in types.

-   **1.3.0**

    -   Add API for search applicant by ID.

-   **1.2.0**

    -   Add search field
    -   Remove required fields.

-   **1.1.0**

    -   First release
